"""User repository backed by TinyDB (data/users.json)."""
from __future__ import annotations

import os
from typing import Optional, Dict, Any

from tinydb import TinyDB, Query


class UserRepository:
    """Minimal repository used by AuthService for login and seeding."""

    def __init__(self, db_path: str = "data/users.json") -> None:
        os.makedirs(os.path.dirname(db_path), exist_ok=True)
        self.db = TinyDB(db_path)
        # Keep users in a named table to avoid collisions if the JSON is reused
        self.table = self.db.table("users")

    # -------- helpers --------
    def _next_id(self) -> int:
        rows = self.table.all()
        if not rows:
            return 1
        return max((r.get("id") or 0) for r in rows) + 1

    # -------- CRUD used by AuthService --------
    def get_user_by_username(self, username: str) -> Optional[Dict[str, Any]]:
        """Return a user dict by username, or None if not found."""
        q = Query()
        res = self.table.get(q.username == username)
        return dict(res) if res else None

    def create_user(self, user_obj: Any) -> int:
        """
        Insert a user.
        Accepts either a dict or a Pydantic model with .dict().
        Ensures unique username.
        """
        data: Dict[str, Any]
        if hasattr(user_obj, "dict"):
            data = user_obj.dict()
        else:
            data = dict(user_obj)

        username = data.get("username")
        if not username:
            raise ValueError("username is required")

        existing = self.get_user_by_username(username)
        if existing:
            # treat as no-op; return existing id
            return existing["id"]

        if not data.get("id"):
            data["id"] = self._next_id()

        self.table.insert(data)
        return data["id"]

    # Optional helpers (not required by login but useful)
    def get_by_id(self, user_id: int) -> Optional[Dict[str, Any]]:
        q = Query()
        res = self.table.get(q.id == user_id)
        return dict(res) if res else None

    def update_user(self, user_id: int, **fields: Any) -> bool:
        q = Query()
        return bool(self.table.update(fields, q.id == user_id))

    def delete_user(self, user_id: int) -> bool:
        q = Query()
        return bool(self.table.remove(q.id == user_id))
